<?php
namespace MajorMedia\Project\Classes;

use Spatie\Browsershot\Browsershot;

/**
 * Browsershot PDF Generator
 *
 * Wrapper class for Spatie Browsershot library
 * Handles PDF generation with consistent configuration
 */
class BrowsershotPdfGenerator
{
    /**
     * Generate PDF from a rendered Blade view
     *
     * @param string $viewName Blade view name (e.g., 'majormedia.project::quote.quote')
     * @param array $data Data to pass to the view
     * @param array $options PDF generation options
     * @return string PDF binary content
     * @throws \Exception
     */
    public function generateFromView($viewName, $data, $options = [])
    {
        try {
            // Render the Blade view to HTML
            $html = view($viewName, $data)->render();

            // Generate PDF from HTML using Browsershot
            return $this->generateFromHtml($html, $options);

        } catch (\Exception $e) {
            \Log::error("Browsershot PDF generation failed: {$e->getMessage()} at {$e->getFile()}:{$e->getLine()}");
            throw $e;
        }
    }

    /**
     * Generate PDF from HTML string
     *
     * @param string $html HTML content
     * @param array $options PDF generation options
     * @return string PDF binary content
     * @throws \Exception
     */
    public function generateFromHtml($html, $options = [])
    {
        try {
            // Default options
            $defaultOptions = [
                'format' => 'A4',
                'orientation' => 'portrait',
                'margin_top' => 0,
                'margin_right' => 0,
                'margin_bottom' => 0,
                'margin_left' => 0,
                'print_background' => true,
                'wait_until_network_idle' => true,
                'timeout' => 60, // 60 seconds timeout
            ];

            // Merge with provided options
            $options = array_merge($defaultOptions, $options);

            // Create Browsershot instance
            $browsershot = Browsershot::html($html)
                ->format($options['format'])
                ->margins(
                    $options['margin_top'],
                    $options['margin_right'],
                    $options['margin_bottom'],
                    $options['margin_left']
                )
                ->timeout($options['timeout']);

            // Set orientation
            if ($options['orientation'] === 'landscape') {
                $browsershot->landscape();
            }

            // Print background graphics
            if ($options['print_background']) {
                $browsershot->showBackground();
            }

            // Wait for network to be idle (useful for loading external resources)
            if ($options['wait_until_network_idle']) {
                $browsershot->waitUntilNetworkIdle();
            }

            // Additional options
            if (isset($options['scale'])) {
                $browsershot->scale($options['scale']);
            }

            if (isset($options['device_scale_factor'])) {
                $browsershot->deviceScaleFactor($options['device_scale_factor']);
            }

            // Generate and return PDF
            return $browsershot->pdf();

        } catch (\Exception $e) {
            \Log::error("Browsershot HTML to PDF conversion failed: {$e->getMessage()}");
            throw new \Exception("PDF generation failed: " . $e->getMessage());
        }
    }

    /**
     * Save PDF to file
     *
     * @param string $html HTML content
     * @param string $path File path to save PDF
     * @param array $options PDF generation options
     * @return bool Success status
     */
    public function saveFromHtml($html, $path, $options = [])
    {
        try {
            $pdfContent = $this->generateFromHtml($html, $options);

            // Ensure directory exists
            $directory = dirname($path);
            if (!file_exists($directory)) {
                mkdir($directory, 0755, true);
            }

            // Save to file
            file_put_contents($path, $pdfContent);

            return file_exists($path);

        } catch (\Exception $e) {
            \Log::error("Failed to save PDF: {$e->getMessage()}");
            return false;
        }
    }

    /**
     * Save PDF from view to file
     *
     * @param string $viewName Blade view name
     * @param array $data Data to pass to the view
     * @param string $path File path to save PDF
     * @param array $options PDF generation options
     * @return bool Success status
     */
    public function saveFromView($viewName, $data, $path, $options = [])
    {
        try {
            $html = view($viewName, $data)->render();
            return $this->saveFromHtml($html, $path, $options);

        } catch (\Exception $e) {
            \Log::error("Failed to save PDF from view: {$e->getMessage()}");
            return false;
        }
    }
}
