<?php
namespace MajorMedia\Project\Classes;

use MajorMedia\Project\Models\Project;
use MajorMedia\Project\Models\Quote;
use MajorMedia\ToolBox\Models\Settings;

/**
 * Quote Data Provider
 *
 * Prepares all data needed for quote PDF generation
 * This class is library-agnostic and can be used with any PDF generator
 */
class QuoteDataProvider
{
    /**
     * Get all data needed for quote PDF generation
     *
     * @param int $projectId Project ID
     * @param int $frontDesignId Front Design ID (1=template, 2=custom)
     * @param Quote $quote Quote instance
     * @return array Data for PDF template
     */
    public function getData($projectId, $frontDesignId, $quote)
    {
        $project = Project::with([
            'project_type',
            'user',
            'sitemap_pages.category'
        ])->find($projectId);

        if (!$project) {
            throw new \Exception("Project not found");
        }

        // Get pages grouped by category
        $pages = $project->sitemap_pages()->with('category')->get();
        $pagesGrouped = $pages->groupBy(function($page) {
            return $page->category ? $page->category->name : 'Sans catégorie';
        });

        // Get features (synced + custom)
        $syncedFeatures = $project->features()
            ->wherePivot('project_id', $projectId)
            ->get();

        $customFeatures = $project->custom_features()->get();
        $allFeatures = $syncedFeatures->merge($customFeatures);

        // Get front design name
        $frontDesign = \MajorMedia\Project\Models\FrontDesign::find($frontDesignId);
        if (!$frontDesign) {
            throw new \Exception("Front design not found");
        }

        // Get company settings
        $settings = Settings::instance();

        // Get pricing breakdown
        $pricing = $quote->calculateTotalCost($project, $frontDesignId);

        // Determine CMS based on project type
        $cmsName = match((int)$project->type_id) {
            1 => 'PrestaShop',  // E-commerce
            2 => 'WordPress',    // Vitrine
            3 => 'WordPress',    // Landing Page
            default => 'WordPress'
        };

        // Get logo path - use public accessible path for Browsershot
        $logoPath = public_path('themes/majorweb/assets/images/logo.png');
        if (!file_exists($logoPath)) {
            $logoPath = public_path('themes/majorweb/assets/images/logo.jpeg');
        }
        if (!file_exists($logoPath)) {
            $logoPath = plugins_path('majormedia/project/assets/images/logo.png');
        }
        if (!file_exists($logoPath)) {
            $logoPath = plugins_path('majormedia/project/assets/images/logo.jpeg');
        }

        // Convert to URL for Browsershot (it can handle URLs better than file paths)
        if (file_exists($logoPath)) {
            // Use asset() helper to get public URL
            $logoUrl = asset('themes/majorweb/assets/images/logo.png');
            if (!file_exists(public_path('themes/majorweb/assets/images/logo.png'))) {
                $logoUrl = asset('themes/majorweb/assets/images/logo.jpeg');
            }
        } else {
            // Fallback to a transparent 1x1 PNG data URI
            $logoUrl = 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNk+M9QDwADhgGAWjR9awAAAABJRU5ErkJggg==';
        }

        $data = [
            'quote' => $quote,
            'project' => $project,
            'pages' => $pages,
            'pagesGrouped' => $pagesGrouped,
            'features' => $allFeatures,
            'front_design' => $frontDesign,
            'cms_name' => $cmsName,
            'pricing' => $pricing,
            'settings' => $settings,
            'date' => now()->format('d/m/Y'),
            'logoPath' => $logoUrl,
            'company' => [
                'name' => 'MAJOR MEDIA',
                'legal' => 'Sté MAJOR MEDIA SARL au capital de 100 000 Dhs',
                'address' => 'Borj Dlalat, Immeuble F, 4 ème & 5 ème Etage',
                'city' => 'Agadir 80000 - Maroc',
                'phone' => '+212 6 64 71 30 37',
                'email' => 'contact@majormedia.ma',
                'website' => 'www.majormedia.ma',
                'bank_name' => 'BANK OF AFRICA',
                'bank_account' => 'MAJOR MEDIA SARL AU',
                'rib' => '0110 1000 0026 2100 0013 3087'
            ]
        ];

        return $data;
    }
}
