<?php
namespace MajorMedia\Project\Classes;

use MajorMedia\Project\Models\Project;
use MajorMedia\Project\Models\Quote;
use MajorMedia\Project\Models\FrontDesign;
use System\Models\File;
use Carbon\Carbon;

/**
 * Quote Generator Service
 *
 * Orchestrates quote generation using Browsershot
 * Main service class that coordinates data preparation, PDF generation, and storage
 */
class QuoteGenerator
{
    protected $dataProvider;
    protected $pdfGenerator;

    public function __construct()
    {
        $this->dataProvider = new QuoteDataProvider();
        $this->pdfGenerator = new BrowsershotPdfGenerator();
    }

    /**
     * Generate quotes for a project (2 quotes: template + custom)
     *
     * @param Project $project
     * @return array Array of Quote instances
     */
    public function generateQuotesForProject(Project $project)
    {
        $quotes = [];

        // Get all front designs (1=template, 2=custom)
        $frontDesigns = FrontDesign::where('is_active', 1)->get();

        if ($frontDesigns->isEmpty()) {
            \Log::warning("No active front designs found. Cannot generate quotes.");
            return $quotes;
        }

        foreach ($frontDesigns as $frontDesign) {
            try {
                $quote = $this->generateQuote($project, $frontDesign->id);
                if ($quote) {
                    $quotes[] = $quote;
                    \Log::info("✅ Quote {$quote->id} generated successfully for design {$frontDesign->name}");
                } else {
                    \Log::error("❌ Failed to generate quote for project {$project->id}, design {$frontDesign->id}");
                }
            } catch (\Exception $e) {
                \Log::error("❌ Exception generating quote for design {$frontDesign->id}: {$e->getMessage()}");
            }
        }

        return $quotes;
    }

    /**
     * Generate a single quote for a project and front design
     *
     * @param Project $project
     * @param int $frontDesignId
     * @return Quote|null
     */
    public function generateQuote(Project $project, int $frontDesignId)
    {
        try {
            \Log::info("🚀 Starting quote generation for project {$project->id}, design {$frontDesignId}");

            // Create or update quote record
            $quote = Quote::firstOrNew([
                'project_id' => $project->id,
                'front_design_id' => $frontDesignId
            ]);

            // Generate reference number if new
            if (!$quote->reference) {
                $quote->reference = $this->generateReference($project);
            }

            // Set client info
            $quote->client_name = $project->user->name ?? $project->name;
            $quote->quote_date = Carbon::now();
            $quote->environment_id = $project->environment_id ?? null;

            // Calculate pricing
            $pricing = $quote->calculateTotalCost($project, $frontDesignId);
            $quote->recomanded_amount = $pricing['total_ht'];
            $quote->final_amount = $pricing['total_ttc'];

            // Save quote first to get ID
            $quote->save();
            \Log::info("💾 Quote record saved with ID: {$quote->id}");

            // Generate PDF
            $pdfFile = $this->generatePdf($project, $frontDesignId, $quote);

            if ($pdfFile) {
                // Attach PDF to quote
                $quote->quote()->add($pdfFile);
                $quote->save();
                \Log::info("✅ Quote {$quote->id} generated with PDF file {$pdfFile->id}");
            } else {
                \Log::error("❌ Quote {$quote->id} saved without PDF - check logs above");
            }

            return $quote;

        } catch (\Exception $e) {
            \Log::error("❌ Quote generation failed for project {$project->id}: {$e->getMessage()} at {$e->getFile()}:{$e->getLine()}");
            return null;
        }
    }

    /**
     * Generate PDF file for quote using Browsershot
     *
     * @param Project $project
     * @param int $frontDesignId
     * @param Quote $quote
     * @return File|null
     */
    protected function generatePdf(Project $project, int $frontDesignId, Quote $quote)
    {
        try {
            \Log::info("📄 Preparing data for PDF generation...");

            // Get data for PDF using data provider
            $data = $this->dataProvider->getData($project->id, $frontDesignId, $quote);

            \Log::info("🎨 Rendering Blade view...");

            // Generate PDF using Browsershot
            $pdfContent = $this->pdfGenerator->generateFromView(
                'majormedia.project::quote.quote',
                $data,
                [
                    'format' => 'A4',
                    'orientation' => 'portrait',
                    'margin_top' => 0,
                    'margin_right' => 0,
                    'margin_bottom' => 0,
                    'margin_left' => 0,
                    'print_background' => true,
                    'wait_until_network_idle' => true,
                    'timeout' => 60
                ]
            );

            \Log::info("📦 PDF generated, size: " . strlen($pdfContent) . " bytes");

            // Generate filename
            $frontDesignName = FrontDesign::find($frontDesignId)->name ?? 'design';
            $frontDesignSlug = str_replace(' ', '_', strtolower($frontDesignName));
            $filename = sprintf(
                'devis_%s_%s_%s.pdf',
                $project->id,
                $frontDesignSlug,
                Carbon::now()->format('Ymd_His')
            );

            // Save to temporary file
            $tempPath = temp_path($filename);
            file_put_contents($tempPath, $pdfContent);

            \Log::info("💾 PDF saved to temp: {$tempPath}");

            // Verify file exists
            if (!file_exists($tempPath)) {
                \Log::error("❌ PDF temp file does not exist at {$tempPath}");
                return null;
            }

            // Create File model instance using October CMS fromFile method
            $file = (new File())->fromFile($tempPath);
            $file->is_public = true;
            $file->save();

            \Log::info("✅ File model created with ID: {$file->id}");

            // Clean up temp file
            @unlink($tempPath);

            return $file;

        } catch (\Exception $e) {
            \Log::error("❌ PDF generation failed: {$e->getMessage()} at {$e->getFile()}:{$e->getLine()}");
            \Log::error("Stack trace: " . $e->getTraceAsString());
            return null;
        }
    }

    /**
     * Generate unique reference number for quote
     *
     * Format: DV-YYYYMM-XXXX (e.g., DV-202601-0001)
     *
     * @param Project $project
     * @return string
     */
    protected function generateReference(Project $project)
    {
        $year = Carbon::now()->format('Y');
        $month = Carbon::now()->format('m');

        // Count existing quotes this month
        $count = Quote::whereYear('created_at', $year)
            ->whereMonth('created_at', $month)
            ->count() + 1;

        return sprintf('DV-%s%s-%04d', $year, $month, $count);
    }
}
