<?php
namespace MajorMedia\Project\Models;

use Model;
use System\Models\File;
use RainLab\User\Models\User;
use MajorMedia\Structure\Models\SiteMap;
use MajorMedia\Project\Models\ProjectType;
use MajorMedia\ToolBox\Traits\ModelAliases;
use MajorMedia\Project\Models\SocialPlatform;
/**
 * Model
 */
class Project extends Model
{
    use \October\Rain\Database\Traits\Validation, ModelAliases;


    // Status (Statut)
    const LABEL_STATUS_1 = 'Brouillon';
    const LABEL_STATUS_2 = 'Completer';

    // Sales Method (Méthode de vente)
    const LABEL_SALES_METHOD_1 = 'Vente en ligne';
    const LABEL_SALES_METHOD_2 = 'Demande de devis';
    const LABEL_SALES_METHOD_3 = 'Catalogue';

    // Delivery Method (Méthode de livraison)
    const LABEL_DELIVERY_METHOD_1 = 'Gérée par le siteweb';
    const LABEL_DELIVERY_METHOD_2 = 'Gérée hors siteweb';



    public $table = 'majormedia_project_projects';

    public $rules = [
    ];
    protected $casts = [
        'is_logo_required' => 'boolean',
        'is_multi_lang' => 'boolean',
        'is_content_required' => 'boolean',
        'languages' => 'array',
        'social_platforms' => 'array',
    ];
    protected $jsonable = [
        'languages',
        'social_platforms',
    ];

    protected $appends = [
        'logo_url',
        'products_content_url',
        'products_images_url',
        'categories_content_url',
        'categories_images_url',
        'brands_content_url',
        'brands_images_url',
        'legal_document_url',
        'privacy_policy_url',
        'terms_and_conditions_url',
        'payment_methods',
        'shipping_parameters',
        'socialPlatforms'
    ];

    protected $fillable = [
        'name',
        'domain',
        'is_logo_required',
        'is_multi_lang',
        'is_content_required',
        'country_name',
        'languages',
        'address',
        'email',
        'phone',
        'working_hour',
        'social_platforms',
        'type_id',
        'user_id',
        'nbr_products',
        'nbr_categories',
        'nbr_brands',
        'sales_method',
        'carrier',
        'delivery_method',
        'shipping_cost_type',
    ];
    protected $visible = [
        'id',
        'name',
        'domain',
        'type_id',
        'is_logo_required',
        'is_multi_lang',
        'is_content_required',
        'country_name',
        'languages',
        'address',
        'email',
        'phone',
        'working_hour',
        'socialPlatforms',
        'shipping_parameters',
        'payment_methods',
        'logo_url',
        'products_content_url',
        'products_images_url',
        'categories_content_url',
        'categories_images_url',
        'brands_content_url',
        'brands_images_url',
        'legal_document_url',
        'privacy_policy_url',
        'terms_and_conditions_url',
        'nbr_products',
        'nbr_categories',
        'nbr_brands',
        'sales_method',
        'carrier',
        'delivery_method',
        'payment_methods',
        'shipping_cost_type',
    ];

    public $hasMany = [
        'sitemaps' => [
            SiteMap::class,
            'key' => 'project_id',
        ],
        ''
    ];

    public $belongsTo = [
        'project_type' => [
            ProjectType::class,
            'key' => 'type_id',
        ],
        'user' => [
            User::class,
            'key' => 'user_id',
        ],
    ];
    public $belongsToMany = [
        'socialPlatforms' => [
            SocialPlatform::class,
            'table' => 'majormedia_project_social_links',
            'key' => 'project_id',
            'otherKey' => 'social_id',
            'pivot' => ['url'],
        ],
        'payment_methods' => [
            Payment::class,
            'table' => 'majormedia_project_project_payment_methods',
            'key' => 'project_id',
            'otherKey' => 'payment_method',
        ],
        'shipping_parameters' => [
            ShippingParameter::class,
            'table' => 'majormedia_project_project_shipping_parameters',
            'key' => 'project_id',
            'otherKey' => 'shipping_parameter_id',
        ],
    ];
    public $attachOne = [
        'logo' => File::class,
        'products_content' => [File::class],
        'products_images' => [File::class],
        'categories_content' => [File::class],
        'categories_images' => [File::class],
        'brands_content' => [File::class],
        'brands_images' => [File::class],
        'legal_document' => [File::class],
        'privacy_policy' => [File::class],
        'terms_and_conditions' => [File::class],

    ];
    public function getLogoUrlAttribute()
    {
        try {
            return $this->logo ? $this->logo->path : null;
        } catch (\Exception $e) {
            \Log::error('Error retrieving logo URL: ' . $e->getMessage());
            return null;
        }
    }
    public function getProductsContentUrlAttribute()
    {
        try {
            return $this->products_content ? $this->products_content->path : null;
        } catch (\Exception $e) {
            \Log::error('Error retrieving products content URL: ' . $e->getMessage());
            return null;
        }
    }

    public function getProductsImagesUrlAttribute()
    {
        try {
            return $this->products_images ? $this->products_images->path : null;
        } catch (\Exception $e) {
            \Log::error('Error retrieving products images URL: ' . $e->getMessage());
            return null;
        }
    }

    public function getCategoriesContentUrlAttribute()
    {
        try {
            return $this->categories_content ? $this->categories_content->path : null;
        } catch (\Exception $e) {
            \Log::error('Error retrieving categories content URL: ' . $e->getMessage());
            return null;
        }
    }

    public function getCategoriesImagesUrlAttribute()
    {
        try {
            return $this->categories_images ? $this->categories_images->path : null;
        } catch (\Exception $e) {
            \Log::error('Error retrieving categories images URL: ' . $e->getMessage());
            return null;
        }
    }

    public function getBrandsContentUrlAttribute()
    {
        try {
            return $this->brands_content ? $this->brands_content->path : null;
        } catch (\Exception $e) {
            \Log::error('Error retrieving brands content URL: ' . $e->getMessage());
            return null;
        }
    }

    public function getBrandsImagesUrlAttribute()
    {
        try {
            return $this->brands_images ? $this->brands_images->path : null;
        } catch (\Exception $e) {
            \Log::error('Error retrieving brands images URL: ' . $e->getMessage());
            return null;
        }
    }

    public function getLegalDocumentUrlAttribute()
    {
        try {
            return $this->legal_document ? $this->legal_document->path : null;
        } catch (\Exception $e) {
            \Log::error('Error retrieving legal document URL: ' . $e->getMessage());
            return null;
        }
    }

    public function getPrivacyPolicyUrlAttribute()
    {
        try {
            return $this->privacy_policy ? $this->privacy_policy->path : null;
        } catch (\Exception $e) {
            \Log::error('Error retrieving privacy policy URL: ' . $e->getMessage());
            return null;
        }
    }

    public function getTermsAndConditionsUrlAttribute()
    {
        try {
            return $this->terms_and_conditions ? $this->terms_and_conditions->path : null;
        } catch (\Exception $e) {
            \Log::error('Error retrieving terms and conditions URL: ' . $e->getMessage());
            return null;
        }
    }

    public function getStatusOptions()
    {
        return $this->guessOptions('LABEL_STATUS_');
    }
    public function getSalesMethodOptions()
    {
        return $this->guessOptions('LABEL_SALES_METHOD_');
    }

    public function getDeliveryMethodOptions()
    {
        return $this->guessOptions('LABEL_DELIVERY_METHOD_');
    }


    // Alias for Sales Method
    public function getSalesMethodAliasAttribute()
    {
        $options = $this->getSalesMethodOptions();
        return isset($options[$this->sales_method]) ? $options[$this->sales_method] : '-';
    }

    // Alias for Delivery Method
    public function getDeliveryMethodAliasAttribute()
    {
        $options = $this->getDeliveryMethodOptions();
        return isset($options[$this->delivery_method]) ? $options[$this->delivery_method] : '-';
    }

    public function getStatusAliasAttribute()
    {
        $options = $this->getStatusOptions();
        return (isset($options[$this->status]) ? $options[$this->status] : '-');
    }

    public function getPaymentMethodsAttribute()
    {
        $paymentMethodIds = $this->payment_methods()->pluck('payment_method')->filter();
        if ($paymentMethodIds->isEmpty()) {
            return collect();
        }
        $paymentMethods = Payment::whereIn('id', $paymentMethodIds)->get(['id', 'name']);
        return $paymentMethods;
    }
    public function getShippingParametersAttribute()
    {
        $shippingParameter = $this->shipping_parameters()->pluck('shipping_parameter_id')->filter();
        if ($shippingParameter->isEmpty()) {
            return collect();
        }
        $shippingParameter = ShippingParameter::whereIn('id', $shippingParameter)->get(['id', 'name']);
        return $shippingParameter;
    }
    public function getSocialPlatformsAttribute()
    {
        $socialPlatforms = $this->socialPlatforms()->get(['social_id', 'url'])->filter();

        if ($socialPlatforms->isEmpty()) {
            return collect();
        }

        return $socialPlatforms->map(function ($platform) {
            $social = SocialPlatform::find($platform->social_id, ['id', 'name']);
            return [
                'id' => $social->id ?? null,
                'name' => $social->name ?? null,
                'url' => $platform->url,
            ];
        })->filter();
    }

    /**
     * Mutators to handle empty strings for integer fields
     */
    public function setNbrProductsAttribute($value)
    {
        $this->attributes['nbr_products'] = empty($value) ? null : $value;
    }

    public function setNbrCategoriesAttribute($value)
    {
        $this->attributes['nbr_categories'] = empty($value) ? null : $value;
    }

    public function setNbrBrandsAttribute($value)
    {
        $this->attributes['nbr_brands'] = empty($value) ? null : $value;
    }

    public function setEmailAttribute($value)
    {
        $this->attributes['email'] = empty($value) ? null : $value;
    }

    public function setCarrierAttribute($value)
    {
        $this->attributes['carrier'] = empty($value) ? null : $value;
    }

    /**
     * After update event - Generate quotes when is_content_required is updated
     *
     * Triggers automatic quote generation using Browsershot when the
     * is_content_required field changes from false to true
     */
    public function afterUpdate()
    {
        // Check if is_content_required was changed during this save
        if ($this->wasChanged('is_content_required') && $this->is_content_required) {
            // Generate quotes using QuoteGenerator with Browsershot
            try {
                \Log::info("🚀 Project {$this->id}: is_content_required changed to true, triggering quote generation");

                $generator = new \MajorMedia\Project\Classes\QuoteGenerator();
                $quotes = $generator->generateQuotesForProject($this);

                if (empty($quotes)) {
                    \Flash::error("❌ Aucun devis n'a été généré. Vérifiez les logs.");
                    \Log::warning("⚠️ No quotes generated for project {$this->id}");
                } else {
                    \Flash::success("✅ " . count($quotes) . " devis générés avec succès!");
                    \Log::info("✅ Successfully generated " . count($quotes) . " quotes for project {$this->id}");
                }
            } catch (\Exception $e) {
                \Flash::error("❌ Erreur lors de la génération. Consultez les logs.");
                \Log::error("❌ Quote generation error for project {$this->id}: {$e->getMessage()} at {$e->getFile()}:{$e->getLine()}");
                \Log::error("Stack trace: " . $e->getTraceAsString());
            }
        }
    }

}
