<?php

namespace MajorMedia\Project\Traits;

use MajorMedia\ToolBox\Models\Settings;
use DB;

/**
 * Estimate Project Cost Trait
 *
 * Calculates project pricing based on:
 * - Project type and front design (base price from pivot table)
 * - Number of pages (from pricing intervals in settings)
 * - Features (individual feature prices)
 */
trait EstimateProjectCost {

    /**
     * Calculate total cost for custom features
     * @param array $features Collection of features with price
     * @return float Total features cost
     */
    public function customFeaturesCost($features) {
        $cost = 0;
        foreach($features as $feature){
            $cost += (float) ($feature->price ?? 0);
        }
        return $cost;
    }

    /**
     * Calculate price based on number of pages using intervals from settings
     * @param int $pageCount Number of pages
     * @return float Price for pages
     */
    public function calculateProjectBasedOnPages($pageCount) {
        $settings = Settings::instance();
        $intervals = $settings->pricing_page_intervals ?? [];

        if (empty($intervals)) {
            return 0;
        }

        foreach ($intervals as $interval) {
            $minPages = (int) ($interval['min_pages'] ?? 0);
            $maxPages = (int) ($interval['max_pages'] ?? 999);
            $price = (float) ($interval['price'] ?? 0);

            if ($pageCount >= $minPages && $pageCount <= $maxPages) {
                return $price;
            }
        }

        // If no interval matches, return last interval price
        $lastInterval = end($intervals);
        return (float) ($lastInterval['price'] ?? 0);
    }

    /**
     * Get base price from project_type and front_design pivot table
     * @param int $projectTypeId
     * @param int $frontDesignId
     * @return float Base price
     */
    public function getBasePrice($projectTypeId, $frontDesignId) {
        $result = DB::table('majormedia_project_type_designs')
            ->where('project_type_id', $projectTypeId)
            ->where('front_design_id', $frontDesignId)
            ->value('initial_price');

        return (float) ($result ?? 0);
    }

    /**
     * Calculate total project cost (HT)
     * @param object $project Project instance
     * @param int $frontDesignId Front design ID (1=template, 2=custom)
     * @return array ['base_price', 'pages_price', 'features_price', 'total_ht', 'tva', 'total_ttc']
     */
    public function calculateTotalCost($project, $frontDesignId) {
        $settings = Settings::instance();
        $tvaRate = (float) ($settings->pricing_tva_rate ?? 20);

        // 1. Base price (project_type + front_design)
        $basePrice = $this->getBasePrice($project->type_id, $frontDesignId);

        // 2. Pages price
        $pageCount = $project->sitemap_pages()->distinct()->count();
        $pagesPrice = $this->calculateProjectBasedOnPages($pageCount);

        // 3. Features price (only paid features)
        $features = $project->features()
            ->wherePivot('project_id', $project->id)
            ->get();

        $customFeatures = $project->custom_features()->get();

        $allFeatures = $features->merge($customFeatures);
        $featuresPrice = $this->customFeaturesCost($allFeatures);

        // Calculate totals
        $totalHT = $basePrice + $pagesPrice + $featuresPrice;
        $tva = $totalHT * ($tvaRate / 100);
        $totalTTC = $totalHT + $tva;

        return [
            'base_price' => $basePrice,
            'pages_price' => $pagesPrice,
            'pages_count' => $pageCount,
            'features_price' => $featuresPrice,
            'total_ht' => $totalHT,
            'tva_rate' => $tvaRate,
            'tva' => $tva,
            'total_ttc' => $totalTTC
        ];
    }
}
